$ErrorActionPreference = "Stop"
$LogFile = "C:\temp\Medicus_Uninstall_Log.txt"
$UninstallFlag = "C:\temp\medicus_uninstall_in_progress.flag"

# Prevent uninstall loop
if (Test-Path $UninstallFlag) {
    Write-Output "Uninstall already in progress. Exiting to prevent loop." | Out-File -Append $LogFile
    exit 0
}
New-Item $UninstallFlag -ItemType File -Force | Out-Null

# Ensure the temp directory exists
New-Item -Path "C:\temp" -ItemType Directory -Force | Out-Null

"--- Medicus Uninstall Start ---" | Out-File -Append $LogFile
Write-Output "Starting Medicus Desktop App uninstallation..."

# Step 0: Force stop any running Medicus processes
Write-Output "Terminating running Medicus processes..."
"Terminating running Medicus processes..." | Out-File -Append $LogFile
$RemainingProcesses = @()
$ProcessNames = @("Medicus Desktop", "com.medicus.dtapp")

foreach ($ProcessName in $ProcessNames) {
    try {
        taskkill /IM "$ProcessName.exe" /F /T 2> $null
    } catch {
        Write-Output "taskkill failed for $ProcessName.exe, continuing..."
        "taskkill failed for $ProcessName.exe, continuing..." | Out-File -Append $LogFile
    }

    try {
        $Remaining = Get-Process -Name $ProcessName -ErrorAction SilentlyContinue
        if ($Remaining) {
            $RemainingProcesses += $ProcessName
            "WARNING: Still running - $ProcessName" | Out-File -Append $LogFile
        } else {
            "Process gone: $ProcessName" | Out-File -Append $LogFile
        }
    } catch {
        "Get-Process failed for ${ProcessName}: $_" | Out-File -Append $LogFile
    }
}

# Step 1: Remove the Appx package for all users
"Removing Appx packages..." | Out-File -Append $LogFile
Write-Output "Removing Appx packages..."
$RemainingPackages = $false
$MaxRetries = 3

Get-AppxPackage -AllUsers | Where-Object {$_.Name -like "dtapp-medicus*"} | ForEach-Object {
    $Retries = 0
    while ($Retries -lt $MaxRetries) {
        try {
            Remove-AppxPackage -Package $_.PackageFullName -AllUsers -ErrorAction Stop
            "Removed Appx package: $($_.PackageFullName)" | Out-File -Append $LogFile
            break
        } catch {
            $Retries++
            Start-Sleep -Seconds 2
            "Retry $Retries/$MaxRetries failed: $_" | Out-File -Append $LogFile
        }
    }
    if ($Retries -eq $MaxRetries) {
        $RemainingPackages = $true
        "FAILED to remove Appx package after $MaxRetries retries: $($_.PackageFullName)" | Out-File -Append $LogFile
    }
}

# Step 2: Clean up Program Files\WindowsApps
"Cleaning Program Files\WindowsApps..." | Out-File -Append $LogFile
$WindowsAppsPath = "C:\Program Files\WindowsApps"
if (Test-Path $WindowsAppsPath) {
    Get-ChildItem -Path $WindowsAppsPath -Filter "dtapp-medicus*" -Force -Directory | ForEach-Object {
        try {
            takeown /F $_.FullName /A /R /D Y | Out-Null
            icacls $_.FullName /grant Administrators:F /T | Out-Null
            Remove-Item -Path $_.FullName -Recurse -Force -ErrorAction Stop
            "Removed folder: $($_.FullName)" | Out-File -Append $LogFile
        } catch {
            "FAILED to remove folder: $($_.FullName): $_" | Out-File -Append $LogFile
        }
    }
}

# Step 3: Clean up user AppData
"Cleaning user AppData folders..." | Out-File -Append $LogFile
$UserProfiles = Get-ChildItem "C:\Users" -Exclude "Public", "Default"
foreach ($user in $UserProfiles) {
    $PackagePath = "$($user.FullName)\AppData\Local\Packages"
    if (Test-Path $PackagePath) {
        Get-ChildItem -Path $PackagePath -Filter "dtapp-medicus*" -Force -Directory | ForEach-Object {
            try {
                Remove-Item -Path $_.FullName -Recurse -Force -ErrorAction Stop
                "Removed folder: $($_.FullName)" | Out-File -Append $LogFile
            } catch {
                "FAILED to remove folder: $($_.FullName): $_" | Out-File -Append $LogFile
            }
        }
    }
}

# Step 4: Clean registry and startup
"Cleaning registry/startup..." | Out-File -Append $LogFile

try {
    if (Test-Path "HKLM:\Software\Microsoft\Windows\CurrentVersion\Run") {
        Remove-ItemProperty -Path "HKLM:\Software\Microsoft\Windows\CurrentVersion\Run" -Name "MedicusDesktopApp" -ErrorAction Stop
        "Registry run key removed." | Out-File -Append $LogFile
    }
} catch {
    "Registry key not found or failed to remove: $_" | Out-File -Append $LogFile
}

$StartupShortcut = "C:\ProgramData\Microsoft\Windows\Start Menu\Programs\Startup\Medicus Desktop.lnk"
if (Test-Path $StartupShortcut) {
    try {
        Remove-Item $StartupShortcut -Force -ErrorAction Stop
        "Removed startup shortcut: $StartupShortcut" | Out-File -Append $LogFile
    } catch {
        "FAILED to remove startup shortcut: $_" | Out-File -Append $LogFile
    }
} else {
    "No shortcut to remove." | Out-File -Append $LogFile
}

# Step 5: Final state
$NeedsReboot = $RemainingProcesses.Count -gt 0 -or $RemainingPackages

if ($NeedsReboot) {
    Write-Output "WARNING: Reboot required to complete uninstall."
    "REBOOT REQUIRED to complete uninstall." | Out-File -Append $LogFile
} else {
    Write-Output "Uninstallation completed successfully."
    "Uninstallation completed successfully." | Out-File -Append $LogFile
}

"--- Medicus Uninstall Complete ---" | Out-File -Append $LogFile
Remove-Item $UninstallFlag -Force -ErrorAction SilentlyContinue
