# Install_Medicus_Desktop_App.ps1
$ErrorActionPreference = "Stop"
$LogFile = "C:\temp\Medicus_Install_Log.txt"

# Create log directory
New-Item -Path "C:\temp" -ItemType Directory -Force | Out-Null
"--- Medicus Install Start @ $(Get-Date -Format o) ---" | Out-File -FilePath $LogFile -Append
"User: $env:USERNAME / $([System.Security.Principal.WindowsIdentity]::GetCurrent().Name)" | Out-File -Append $LogFile

# Script location and MSIX
$ScriptPath = $MyInvocation.MyCommand.Definition
"Script: $ScriptPath" | Out-File -Append $LogFile
$ScriptDir = Split-Path -Parent $ScriptPath
$MSIX = Get-ChildItem -Path $ScriptDir -Filter *.msix | Select-Object -First 1
if (-not $MSIX) {
    "MSIX not found in: $ScriptDir" | Out-File -Append $LogFile
    Write-Output "ERROR: MSIX package missing."
    exit 1
}
"MSIX: $($MSIX.FullName)" | Out-File -Append $LogFile
"TEMP: $env:TEMP" | Out-File -Append $LogFile

# Check if already installed for this user
$IsInstalled = Get-AppxPackage | Where-Object { $_.Name -like "dtapp-medicus*" }
if ($IsInstalled) {
    "--- Medicus already installed. Skipping provision." | Out-File -Append $LogFile
    Write-Output "Medicus Desktop already installed. Skipping."
    exit 0
}

# Provision MSIX package
try {
    Add-AppProvisionedPackage -Online -PackagePath $MSIX.FullName -SkipLicense | Out-Null
    "Provisioned: OK" | Out-File -Append $LogFile
} catch {
    "Provision FAILED: $_" | Out-File -Append $LogFile
    Write-Output "ERROR: Package install failed."
    exit 1
}

Start-Sleep -Seconds 5

# Detect AUMID
$AUMID = (Get-StartApps | Where-Object { $_.AppId -like "*dtapp-medicus*" } | Select-Object -First 1).AppId
if (-not $AUMID) {
    # Fallback: build manually
    $Appx = Get-AppxPackage -AllUsers | Where-Object { $_.Name -like "dtapp-medicus*" } | Select-Object -First 1
    if ($Appx) {
        $AUMID = "$($Appx.Name)_$($Appx.PublisherId)!App"
        "AUMID fallback: $AUMID" | Out-File -Append $LogFile
    } else {
        "AUMID not detected." | Out-File -Append $LogFile
        Write-Output "ERROR: App ID not found."
        exit 1
    }
} else {
    "AUMID: $AUMID" | Out-File -Append $LogFile
}

# Launch once to trigger registration
try {
    Start-Process "explorer.exe" -ArgumentList "shell:AppsFolder\$AUMID"
    Start-Sleep -Seconds 8
    "Launch OK" | Out-File -Append $LogFile
} catch {
    "Launch FAIL: $_" | Out-File -Append $LogFile
}

# Add shortcut in all-users Startup folder
$WScriptShell = New-Object -ComObject WScript.Shell
$StartupPath = "$env:ProgramData\Microsoft\Windows\Start Menu\Programs\Startup"
$ShortcutPath = Join-Path $StartupPath "Medicus Desktop.lnk"
$StartupShortcutSuccess = $false

try {
    $Shortcut = $WScriptShell.CreateShortcut($ShortcutPath)
    $Shortcut.TargetPath = "explorer.exe"
    $Shortcut.Arguments = "shell:AppsFolder\$AUMID"
    $Shortcut.Save()
    if (Test-Path $ShortcutPath) {
        $StartupShortcutSuccess = $true
        "Shortcut OK: $ShortcutPath" | Out-File -Append $LogFile
    } else {
        "Shortcut missing after save." | Out-File -Append $LogFile
    }
} catch {
    "Shortcut FAIL: $_" | Out-File -Append $LogFile
}

# Fallback to registry if shortcut failed
if (-not $StartupShortcutSuccess) {
    try {
        $Command = "cmd /c start shell:AppsFolder\$AUMID"
        Set-ItemProperty -Path "HKLM:\Software\Microsoft\Windows\CurrentVersion\Run" -Name "MedicusDesktopApp" -Value $Command
        "Fallback: Registry added" | Out-File -Append $LogFile
    } catch {
        "Fallback FAIL: $_" | Out-File -Append $LogFile
    }
} else {
    "Fallback not needed" | Out-File -Append $LogFile
}

"--- Medicus Install Complete ---" | Out-File -Append $LogFile
Write-Output "Medicus Desktop App installed. Log: $LogFile"
